import os, uos, sys, machine, network
import time

# -----------------------------
# Command Registry
# -----------------------------
CMDS = {}

def command(name, help_txt=""):
    def decorator(func):
        CMDS[name] = {"func": func, "help": help_txt}
        return func
    return decorator

# -----------------------------
# Filesystem Commands
# -----------------------------
@command("ls", "List directory contents")
def ls(args):
    path = args[0] if args else "."
    try:
        for f in os.listdir(path):
            print(f)
    except:
        print("Error: cannot access", path)

@command("cd", "Change directory")
def cd(args):
    path = args[0] if args else "/"
    try: os.chdir(path)
    except: print("Error: cannot change to", path)

@command("pwd", "Print working directory")
def pwd(args): print(os.getcwd())

@command("df", "Show disk usage")
def df(args):
    import uos
    try:
        stat = uos.statvfs("/")
        total = stat[0]*stat[2]/1024
        free = stat[0]*stat[3]/1024
        print(f"Total: {total:.1f}KB Free: {free:.1f}KB")
    except: print("df not supported")

@command("man", "Show man page")
def man(args):
    if not args: print("Usage: man <command>"); return
    file = f"/etc/man/{args[0]}.txt"
    if os.path.exists(file):
        with open(file) as f: print(f.read())
    else: print("No manual entry for", args[0])

# -----------------------------
# WiFi Commands
# -----------------------------
@command("wifi", "Manage WiFi: scan/connect")
def wifi_cmd(args):
    wlan = network.WLAN()
    wlan.active(True)
    if not args:
        print("Usage: wifi scan/connect <SSID> [password]")
        return
    if args[0] == "scan":
        nets = wlan.scan()
        for i,n in enumerate(nets):
            sec = "Open" if n[4]==0 else "Secured"
            print(f"[{i}] {n[0].decode()} ({sec})")
    elif args[0] == "connect":
        ssid = args[1]
        passwd = args[2] if len(args)>2 else ""
        wlan.connect(ssid, passwd)
        while wlan.status() == network.STAT_CONNECTING: machine.idle()
        if wlan.isconnected(): print("Connected")
        else: print("Failed")

# -----------------------------
# BLE Command Placeholder
# -----------------------------
@command("ble", "Manage BLE devices")
def ble_cmd(args):
    print("BLE commands not yet implemented (stub)")

# -----------------------------
# Run Python Scripts / .fos apps
# -----------------------------
@command("run", "Run a .py or .fos app")
def run(args):
    if not args: print("Usage: run <file>"); return
    file = args[0]
    if not os.path.exists(file): print("File not found"); return
    if file.endswith(".py"):
        try: exec(open(file).read(), globals())
        except Exception as e: print("Error:", e)
    elif file.endswith(".fos"):
        try:
            # FOS compiled format stub
            with open(file,"rb") as f: code = f.read()
            exec(code, globals())
        except Exception as e: print("Error running .fos:", e)
    else:
        print("Unknown app format:", file)

# -----------------------------
# Shell Loop
# -----------------------------
while True:
    try:
        inp = input(f"{os.getcwd()} $ ")
        if not inp: continue
        parts = inp.split()
        cmd = parts[0]
        args = parts[1:]
        if cmd in CMDS:
            CMDS[cmd]["func"](args)
        else:
            print("Unknown command:", cmd)
    except KeyboardInterrupt:
        print("\nCtrl+C detected")
